/**
 * This file is part of waster-0.1.
 *
 *    waster-0.1 is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    waster-0.1 is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with waster-0.1.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.pimeca.waster;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.regex.Pattern;

import com.pimeca.waster.logging.LoggingHandler;
import com.pimeca.waster.oscmd.AbstractOsCmdLauncher;
import com.pimeca.waster.oscmd.OsCmdLauncher;
import com.pimeca.waster.util.FileProc;
import com.pimeca.waster.util.filefilter.FilterPatternFile;

/**
 * Wrapper for launching a computation with aster 3 main method pre() to set the
 * data launch() to execute computation (inherited method) post() to get the
 * results and clean all necessary files for execution are within
 * ROOT_ASTER/runtime
 * 
 * @author maurel
 * 
 */
public class AsterLauncher extends AbstractOsCmdLauncher implements OsCmdLauncher {
    /**
     * /** aster short version
     */
    private final static String ASTER_SHORT_VERSION = "10.3";
    /**
     * aster STA version
     */
    private final static String STA = "STA" + ASTER_SHORT_VERSION;
    /*
     * runtime dir uri
     */
    private String runtimeDirUri = "file:///C:/aster/" + ASTER_SHORT_VERSION + "/runtime/";
    private String exe = runtimeDirUri + STA + "/waster.exe";
    private String eficasPathDirName = "bibpyt";
    private String eficasPathDirUri = runtimeDirUri + STA + "/" + eficasPathDirName;
    private String toolDirUri = runtimeDirUri + "OUTILS/";
    private String materialDirUri = runtimeDirUri + STA + "/materiau/";
    private String elementCatalogSrcUri = runtimeDirUri + STA + "/catalog/elem.1";
    private String memory = "8";
    /**
     * Map for correspondence between user data files extension and number n in
     * aster data files fort.n
     */
    private Map<String, Integer> extNumberMap = new HashMap<String, Integer>();
    /**
     * List of uris where are located user data files
     */
    private List<String> dataFileUris = new ArrayList<String>();
    private String srcCmdFileDirUri = "";
    private String cmdFileExtension = "comm";
    private String filesRootName = "a";
    private String cmdFileUri = null;
    /**
     * base dir uri (./base) for global and pick
     */
    private String baseDirUri = null;
    /**
     * delete temporary working dir and all its content after execution normally
     * ./work is created before launching execution
     */
    private boolean postDeleteWorkDir = false;
    /**
     * delete base dir and all its content after execution (./base)
     */
    private boolean postDeleteBaseDir = false;

    @SuppressWarnings("unused")
    private AsterLauncher() {
    }

    /**
     * constructor set the extension of command file used set the directory
     * where the user data files are located set the correspondance between user
     * data file extensions and aster data file extensions number
     * 
     * @param cmdFileUri
     *            absolute uri of command file
     */
    public AsterLauncher(String cmdFileUri) {
        this.cmdFileUri = cmdFileUri;
        cmdFileExtension = getExtension(cmdFileUri);
        srcCmdFileDirUri = FileProc.newFile(cmdFileUri).getParentFile().toURI().toString();
        setFilesRootName(cmdFileUri.replaceAll(".*/(.+)\\." + cmdFileExtension, "$1"));
        buildExtNumberMap();
        this.setDefaultBaseDirUri();
    }

    /*
     * create the list of options that will be used for the command line
     */
    @Override
    public List<String> getOptions() {
        if (getWorkingDirUri() == null) {
            throw (new NullPointerException("no working directory has been set"));
        }
        List<String> cmd = new ArrayList<String>();
        cmd.add(FileProc.newFile(exe).getAbsolutePath());
        cmd.add(FileProc.newFile(eficasPathDirUri + eficasPathDirName + "/Execution/E_SUPERV.py").getAbsolutePath());
        cmd.add("-eficas_path");
        cmd.add(FileProc.newFile(eficasPathDirUri + eficasPathDirName).getAbsolutePath());
        cmd.add("-commandes");
        String cmdFilePath = FileProc.newFile(getWorkingDirUri() + "fort.1").getAbsolutePath();
        cmd.add(cmdFilePath);
        cmd.add("-rep_outils");
        cmd.add(FileProc.newFile(toolDirUri).getAbsolutePath());
        cmd.add("-rep_mat");
        cmd.add(FileProc.newFile(materialDirUri).getAbsolutePath());
        cmd.add("-memjeveux");
        cmd.add(memory);
        return cmd;
    }

    /**
     * preparing of data before launching execution
     */
    public void pre() {
        if (getWorkingDirUri() == null) {
            throw (new NullPointerException("no working directory has been set"));
        }
        for (ListIterator<String> it = dataFileUris.listIterator(); it.hasNext();) {
            String srcUri = it.next();
            String extension = getExtension(srcUri);
            StringBuffer sb = new StringBuffer();
            sb.append(".*");
            sb.append(this.filesRootName);
            sb.append("\\d+");
            if (extNumberMap.containsKey(extension)) {
                String fortExtension = String.valueOf(extNumberMap.get(extension));
                String tgtUri = this.getWorkingDirUri() + "fort." + fortExtension;
                FileProc.copyFile(srcUri, tgtUri);
            } else if (srcUri.matches(sb.toString())) {
                String tgtUri = this.getWorkingDirUri() + "fort." + extension;
                FileProc.copyFile(srcUri, tgtUri);
            } else {
                if (!extension.matches("para") && !extension.matches("")) {
                    System.out.println("ignoring file " + srcUri + "...");
                }
            }
        }
        if (isPoursuite()) {
            FilterPatternFile fpf = new FilterPatternFile(".*((pick)|(glob))\\.\\d+");
            File[] files = FileProc.newFile(this.getBaseDirUri()).listFiles(fpf);
            if (files != null) {
                for (int i = 0; i < files.length; i++) {
                    String srcUri = files[i].toURI().toString();
                    String tgtUri = this.getWorkingDirUri() + files[i].getName();
                    FileProc.copyFile(srcUri, tgtUri);
                }
            }
        }
        String srcUri = this.elementCatalogSrcUri;
        String tgtUri = getWorkingDirUri() + "elem.1";
        FileProc.copyFile(srcUri, tgtUri);

        FileProc.createDir(getRepe_outDirUri());
        FileProc.createDir(getRepe_EnsightDirUri());
        FileProc.createDir(getResu_EnsightDirUri());
    }

    /**
     * after execution copying of files from temporary working dir to user dir
     */
    public void post() {
        if (getWorkingDirUri() == null) {
            throw (new NullPointerException("no working directory has been set"));
        }
        String srcUri = this.getWorkingDirUri() + "fort.6";
        String tgtUri = this.srcCmdFileDirUri + this.filesRootName + ".0.mess";
        tgtUri = incrementFileUri(tgtUri);
        copyFile(srcUri, tgtUri);
        srcUri = this.getWorkingDirUri() + "fort.8";
        tgtUri = this.srcCmdFileDirUri + this.filesRootName + ".0.resu";
        tgtUri = incrementFileUri(tgtUri);
        copyFile(srcUri, tgtUri);
        srcUri = this.getWorkingDirUri() + "fort.9";
        tgtUri = this.srcCmdFileDirUri + this.filesRootName + ".0.erre";
        tgtUri = incrementFileUri(tgtUri);
        copyFile(srcUri, tgtUri);
        srcUri = this.getWorkingDirUri() + "fort.80";
        tgtUri = this.srcCmdFileDirUri + this.filesRootName + ".0.med";
        tgtUri = incrementFileUri(tgtUri);
        copyFile(srcUri, tgtUri);
        srcUri = this.getWorkingDirUri() + "fort.37";
        tgtUri = this.srcCmdFileDirUri + this.filesRootName + ".cast";
        copyFile(srcUri, tgtUri);
        srcUri = this.getWorkingDirUri() + "glob.1";
        tgtUri = this.srcCmdFileDirUri + "base/glob.1";
        copyFile(srcUri, tgtUri);
        srcUri = this.getWorkingDirUri() + "pick.1";
        tgtUri = this.srcCmdFileDirUri + "base/pick.1";
        copyFile(srcUri, tgtUri);
        if (this.postDeleteWorkDir) {
            if (FileProc.newFile(getWorkingDirUri()).exists()) {
                FileProc.deleteDir(this.getWorkingDirUri());
            }
        }
        if (this.postDeleteBaseDir) {
            if (getBaseDirUri() == null) {
                LoggingHandler.getLogger().info("cannot delete base dir uri:" + getBaseDirUri());
            } else {
                if (FileProc.newFile(getBaseDirUri()).exists()) {
                    FileProc.deleteDir(getBaseDirUri());
                }
            }
        }
    }

    private void copyFile(String srcUri, String tgtUri) {
        if (FileProc.newFile(srcUri).exists()) {
            FileProc.copyFile(srcUri, tgtUri);
        }
    }

    private void buildExtNumberMap() {
        this.extNumberMap.put(cmdFileExtension, Integer.valueOf(1));
        this.extNumberMap.put("mgib", Integer.valueOf(19));
        this.extNumberMap.put("msup", Integer.valueOf(19));
        this.extNumberMap.put("msh", Integer.valueOf(19));
        this.extNumberMap.put("mail", Integer.valueOf(20));
        this.extNumberMap.put("mmed", Integer.valueOf(20));
    }

    /**
     * add a file to the list of user data files
     * 
     * @param dataFileUri
     *            (absolute uri)
     */
    public void addDataFileUri(String dataFileUri) {
        this.dataFileUris.add(dataFileUri);
    }

    private String getExtension(String uri) {
        String extension = (uri.matches(".*\\.([^/]+)")) ? uri.replaceAll(".*\\.([^/]+)", "$1") : "";
        return extension;
    }

    /**
     * set root name root name is the common name of user data files example for
     * root name a: a.comm, a.mgib,...
     * 
     * @param filesRootName
     */
    public void setFilesRootName(String filesRootName) {
        this.filesRootName = filesRootName;
    }

    /**
     * set the flag to allow for deleting of temporary work directory true
     * allows deletion
     * 
     * @param postDeleteWorkDir
     */
    public void setPostDeleteWorkDir(boolean postDeleteWorkDir) {
        this.postDeleteWorkDir = postDeleteWorkDir;
    }

    /**
     * @param postDeleteBaseDir
     *            if true will delete base dir (global, pick)
     */
    public void setPostDeleteBaseDir(boolean postDeleteBaseDir) {
        this.postDeleteBaseDir = postDeleteBaseDir;
    }

    /**
     * @return the default uri used by aster during execution needed when some
     *         tools are used
     */
    public String getRepe_outDirUri() {
        return this.getWorkingDirUri() + "REPE_OUT";
    }

    /**
     * @return the default uri used by aster during execution needed when
     *         ensight is used
     */
    public String getRepe_EnsightDirUri() {
        return this.getWorkingDirUri() + "REPE_ENSIGHT";
    }

    /**
     * @return the default uri used by aster during execution needed when
     *         ensight is used
     */
    public String getResu_EnsightDirUri() {
        return this.getWorkingDirUri() + "RESU_ENSIGHT";
    }

    /**
     * set default temporary working dir to: dir_of_cmd_file/work
     */
    public void setDefaultWorkingDirUri() {
        setWorkingDirUri(this.srcCmdFileDirUri + "work/");
    }

    /**
     * @return absolute uri of command file
     */
    public String getSrcCmdFileDirUri() {
        return this.srcCmdFileDirUri;
    }

    /**
     * create default list of user data files including glob.n and pick.n if
     * they exist
     * 
     */
    public void setDefaultDataFiles() {
        FilterPatternFile fpf = new FilterPatternFile(".*/" + filesRootName + "\\..+");
        File[] dataFiles = FileProc.newFile(srcCmdFileDirUri).listFiles(fpf);
        for (int i = 0; i < dataFiles.length; i++) {
            this.dataFileUris.add(dataFiles[i].toURI().toString());
        }
        fpf = new FilterPatternFile(".*((glob)|(pick))\\.\\d+");
        if (FileProc.newFile(srcCmdFileDirUri + "base").exists()) {
            dataFiles = FileProc.newFile(srcCmdFileDirUri + "base").listFiles(fpf);
            for (int i = 0; i < dataFiles.length; i++) {
                this.dataFileUris.add(dataFiles[i].toURI().toString());
            }
        }
    }

    /**
     * set the value used for aster cmd line memjeveux (see aster doc: value in
     * words)
     * 
     * @param memory
     */
    public void setMemory(String memory) {
        if (memory.matches("\\d+")) {
            this.memory = memory;
        } else {
            throw (new IllegalArgumentException("no numeric value found in: " + memory));
        }
    }

    /**
     * set runtime directory where all files needed for aster execution are
     * 
     * @param runtimeDirUri
     *            (absolute uri)
     */
    public void setRuntimeDirUri(String runtimeDirUri) {
        this.runtimeDirUri = runtimeDirUri;
        eficasPathDirUri = runtimeDirUri + STA + "/";
        toolDirUri = runtimeDirUri + "OUTILS/";
        materialDirUri = runtimeDirUri + STA + "/materiau/";
        elementCatalogSrcUri = runtimeDirUri + STA + "/catalog/elem.1";
    }

    /**
     * adding .n to the file uri to avoid duplicate names dir/name.ext becomes
     * dir/name.n.ext where n is incremented from existing file
     * 
     * @param uri
     *            (absolute uri)
     * @return uri (absolute modified uri)
     */
    private String incrementFileUri(String uri) {
        String regex = "(.*/.+\\.)\\d+(\\.[a-z0-9]+)";
        if (!uri.matches(regex)) {
            throw (new IllegalArgumentException("no match found for" + uri));
        }
        int n = 0;
        while (FileProc.newFile(uri).exists()) {
            n++;
            uri = uri.replaceAll(regex, "$1" + n + "$2");
        }

        return uri;
    }

    /**
     * @param debug
     *            true to generate debug executable
     */
    public void setExe(boolean debug) {
        String dir = runtimeDirUri + STA + "/";
        exe = dir + ((debug) ? "wasterd.exe" : "waster.exe");
    }

    /**
     * @return the baseDirUri
     */
    public String getBaseDirUri() {
        return baseDirUri;
    }

    /**
     * @param baseDirUri
     *            the baseDirUri to set
     */
    public void setBaseDirUri(String baseDirUri) {
        this.baseDirUri = baseDirUri;
    }

    public void setDefaultBaseDirUri() {
        setBaseDirUri(this.srcCmdFileDirUri + "base/");
    }

    private boolean isPoursuite() {
        Pattern p = Pattern.compile(".*[^#]POURSUITE\\(.*\\).*", Pattern.DOTALL);
        return p.matcher(FileProc.getString(FileProc.newFile(this.cmdFileUri))).matches();
    }
}
