/**
 * This file is part of waster-0.1.
 *
 *    waster-0.1 is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    waster-0.1 is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with waster-0.1.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.pimeca.waster;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.cli.PosixParser;

import com.pimeca.waster.logging.LoggingHandler;

/**
 * wrapper for a command line with GNUParser Appache commons-cli
 * 
 * @author maurel
 * 
 */
public class CmdLineHandler {
    private final int ERROR_EXIT_VALUE = 1;
    private CommandLine cmdLine = null;

    @SuppressWarnings("unused")
    private CmdLineHandler() {
    }

    /**
     * parsing arguments
     * 
     * @param args
     *            command line
     */
    public CmdLineHandler(String[] args) {
        this.cmdLine = parse(getOptions(), args);
    }

    protected CommandLine getCommandLine() {
        return cmdLine;
    }

    /**
     * parse with Posix parser. Print help and exit if aster command file name
     * not found
     * 
     * @param options
     *            command line options
     * @param input
     *            command line input
     * @return parsed command line
     */
    private CommandLine parse(Options options, String[] input) {
        CommandLine cmdLine = null;
        try {
            cmdLine = new PosixParser().parse(getOptions(), input);
            if (cmdLine.getArgs().length == 0) {
                LoggingHandler.getLogger().finer("no aster command file uri found");
                printHelp();
                System.exit(ERROR_EXIT_VALUE);
            }
        } catch (ParseException e) {
            LoggingHandler.getLogger().throwing(CmdLineHandler.class.getName(), "getCommandLine", e);
            e.printStackTrace();
        }
        return cmdLine;
    }

    /**
     * creates org.apache.commons.cli.Options for the command line class
     * 
     * @return options
     */
    private Options getOptions() {
        Options options = new Options();

        Option helpOption = new Option("help", "print this message");
        options.addOption(helpOption);

        OptionBuilder.hasArg();
        OptionBuilder.withArgName("absolute uri");
        OptionBuilder.withDescription("normally set by shell script");
        options.addOption(OptionBuilder.create("defaultruntime"));

        OptionBuilder.withArgName("integer");
        OptionBuilder.hasArg();
        OptionBuilder.withDescription("memory in words");
        options.addOption(OptionBuilder.create("m"));

        OptionBuilder.withDescription("launch debug aster executable if available");
        options.addOption(OptionBuilder.create("d"));

        OptionBuilder.withDescription("clean working directory when finished");
        options.addOption(OptionBuilder.create("c"));

        OptionBuilder.withDescription("clean base directory when finished");
        options.addOption(OptionBuilder.create("b"));

        OptionBuilder.withArgName("directory uri");
        OptionBuilder.hasArg();
        OptionBuilder.withDescription("working directory absolute uri or path (default to cmd_file_uri/work)");
        options.addOption(OptionBuilder.create("w"));

        OptionBuilder.withArgName("absolute uri");
        OptionBuilder.hasArg();
        OptionBuilder.withDescription("runtime dir absolute uri (file:///C:/aster/runtime) or absolute path (ex: C:\\aster\\runtime\\)");
        options.addOption(OptionBuilder.create("r"));

        return options;
    }

    /**
     * print help statement to standard output
     */
    private void printHelp() {
        String footer = "\nwaster-0.1  Copyright (C) 2010  pimeca\nThis program comes with ABSOLUTELY NO WARRANTY;\nThis is free software, and you are welcome to redistribute it under certain conditions; see COPYING for details.\n";
        HelpFormatter hf = new HelpFormatter();
        hf.printHelp("waster [options] foo.comm\n", "", this.getOptions(), footer);
    }
}
