/**
 * This file is part of waster-0.1.
 *
 *    waster-0.1 is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    waster-0.1 is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with waster-0.1.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.pimeca.waster;

import java.io.File;

import org.apache.commons.cli.CommandLine;

import com.pimeca.waster.util.FileProc;

/**
 * wrapper to handle the whole process
 * 
 * @author maurel
 * 
 */
public class InputFileHandler {
    /**
     * value of memjeveux in aster launch command must be a number
     */
    private String memory = "";
    /**
     * absolute uri of aster command file
     */
    private String cmdFileUri = null;
    /**
     * absolute uri of runtime directory where are all files necessary for aster
     * execution
     */
    private String runtimeDirUri = null;
    /**
     * absolute uri of temporary working directory
     */
    private String workDirUri = null;
    /**
     * if not null debug executable is used
     */
    private String debug = null;
    /**
     * command line handler
     */
    private CmdLineHandler cmdLineHandler = null;
    /**
     * if not null delete working dir when finished
     */
    private String cleanWorkingDir = null;
    private String cleanBaseDir = null;

    @SuppressWarnings("unused")
    private InputFileHandler() {
    }

    /**
     * constructor initialising arguments
     * 
     * @param args
     */
    public InputFileHandler(String[] args) {
        this.cmdLineHandler = new CmdLineHandler(args);
    }

    /**
     * launches aster execution
     * 
     * @param cmdFileUri
     *            (abolute uri)
     */
    public void execute(String cmdFileUri) {
        AsterLauncher al = new AsterLauncher(cmdFileUri);
        if (memory.matches("\\d+")) {
            al.setMemory(memory);
        }
        if (cleanWorkingDir == null) {
            al.setPostDeleteWorkDir(false);
        } else {
            al.setPostDeleteWorkDir(true);
        }
        if (cleanBaseDir == null) {
            al.setPostDeleteBaseDir(false);
        } else {
            al.setPostDeleteBaseDir(true);
        }
        if (workDirUri == null) {
            al.setDefaultWorkingDirUri();
        } else {
            al.setWorkingDirUri(this.workDirUri);
        }
        if (runtimeDirUri != null) {
            al.setRuntimeDirUri(this.runtimeDirUri);
        }
        boolean isDebug = (debug != null);
        al.setExe(isDebug);
        al.setDefaultDataFiles();
        al.setFileOutputStream(al.getWorkingDirUri() + "fort.6");
        al.setDefaultBaseDirUri();
        al.pre();
        al.launchCmd(al.getOptions());
        al.post();
    }

    /**
     * launches aster execution with the file command that has been previously
     * setup
     * 
     */
    public void execute() {
        execute(this.cmdFileUri);
    }

    /**
     * setup of this class data
     * 
     */
    public void setOptions() {
        CommandLine cmdLine = this.cmdLineHandler.getCommandLine();
        this.cmdFileUri = checkCmdFileUri(cmdLine.getArgs()[0]);
        if (cmdLine.hasOption("defaultruntime")) {
            runtimeDirUri = checkRuntime(cmdLine.getOptionValue("defaultruntime"));
        }
        if (cmdLine.hasOption("r")) {
            runtimeDirUri = checkRuntime(cmdLine.getOptionValue("r"));
        }
        if (cmdLine.hasOption("w")) {
            workDirUri = cmdLine.getOptionValue("w");
            if (!workDirUri.matches(".*/")) {
                workDirUri = workDirUri + "/";
            }
        }
        if (cmdLine.hasOption("m")) {
            memory = cmdLine.getOptionValue("m");
        }
        if (cmdLine.hasOption("d")) {
            debug = "true";
        }
        if (cmdLine.hasOption("c")) {
            this.cleanWorkingDir = "true";
        }
        if (cmdLine.hasOption("b")) {
            this.cleanBaseDir = "true";
        }
    }

    /**
     * try to transform a windows file path into absolute uri and checks whether
     * the runtime directory exists if candidateRuntimeDirUri is absolute uri
     * only checks whether the file exists
     * 
     * @param candidateRuntimeDirUri
     *            (absolute uri or file path)
     * @return candidateRuntimeDirUri as abolute uri
     */
    private String checkRuntime(String candidateRuntimeDirUri) {
        File f = null;
        if (candidateRuntimeDirUri.matches("file://*/*.*") || candidateRuntimeDirUri.matches("http://.*")) {
            f = FileProc.newFile(candidateRuntimeDirUri);
        } else {
            f = new File(candidateRuntimeDirUri);
        }
        if (f.isDirectory()) {
            candidateRuntimeDirUri = f.toURI().toString();
        } else {
            throw (new IllegalArgumentException("unable to find directory:" + candidateRuntimeDirUri));
        }
        return candidateRuntimeDirUri;
    }

    /**
     * try to transform a windows file path into absolute uri and checks whether
     * the runtime directory exists if candidateCmdFileUri is absolute uri only
     * checks whether the file exists
     * 
     * @param candidateCmdFileUri
     *            (absolute uri or file path)
     * @return candidateCmdFileUri as abolute uri
     */
    private String checkCmdFileUri(String candidateCmdFileUri) {
        File f = null;
        if (candidateCmdFileUri.matches("file://*/*.*") || candidateCmdFileUri.matches("http://.*")) {
            f = FileProc.newFile(candidateCmdFileUri);
        } else {
            f = new File(candidateCmdFileUri);
        }
        if (f.exists()) {
            candidateCmdFileUri = f.toURI().toString();
        } else {
            boolean isComm = candidateCmdFileUri.matches(".*\\.comm");
            candidateCmdFileUri = (!isComm) ? candidateCmdFileUri + ".comm" : candidateCmdFileUri;
            if (candidateCmdFileUri.matches("file://*/*.*") || candidateCmdFileUri.matches("http://.*")) {
                f = FileProc.newFile(candidateCmdFileUri);
            } else {
                f = new File(candidateCmdFileUri);
            }
        }
        if (f.exists()) {
            candidateCmdFileUri = f.toURI().toString();
        } else {
            throw (new IllegalArgumentException("unable to find file:" + f.toURI()));
        }
        return candidateCmdFileUri;
    }
}
