/**
 * This file is part of waster-0.1.
 *
 *    waster-0.1 is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    waster-0.1 is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with waster-0.1.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.pimeca.waster.oscmd;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;

import org.apache.commons.exec.CommandLine;
import org.apache.commons.exec.DefaultExecutor;
import org.apache.commons.exec.ExecuteException;
import org.apache.commons.exec.ExecuteStreamHandler;
import org.apache.commons.exec.ExecuteWatchdog;
import org.apache.commons.exec.PumpStreamHandler;
import org.apache.commons.io.output.TeeOutputStream;

import com.pimeca.waster.logging.LoggingHandler;
import com.pimeca.waster.util.FileProc;

/**
 * abstract class implementing OsCmdLauncher interface
 * 
 * 
 * @author maurel
 * 
 */
public abstract class AbstractOsCmdLauncher implements OsCmdLauncher {
    /**
     * Watchdog time wait in milli-seconds negative value is for no watchdog
     */
    private long timeWait = -1;
    /**
     * list of options for the os command
     */
    private List<String> options = new ArrayList<String>();
    /**
     * working directory for the os command
     */
    private String workingDirUri = null;
    /**
     * file output stream for the os command
     */
    private OutputStream fileOutputStream = null;
    /**
     * true if console output is required
     */
    private boolean isConsoleOutputRequired = true;
    /**
     * flag to delete os working directory before launching the command
     */
    private boolean preDeleteWorkDir = false;
    /**
     * array of exit values that are considered as error if null no value throws
     * exception (see apache.commons.cli)
     */
    private int[] errorExitValues = null;

    /*
     * launches the os command
     * 
     * @see com.pimeca.waster.oscmd.OsCmdLauncher#launchCmd(java.util.List)
     */
    public int launchCmd(List<String> options) {
        CommandLine commandLine = null;
        int exitValue = 999;
        DefaultExecutor executor = null;
        try {
            for (ListIterator<String> it = options.listIterator(); it.hasNext();) {
                if (commandLine == null) {
                    commandLine = new CommandLine(it.next());
                } else {
                    commandLine.addArgument(it.next());
                }
            }
            executor = new DefaultExecutor();
            if (this.workingDirUri != null) {
                executor.setWorkingDirectory(FileProc.newFile(workingDirUri));
            }
            TeeOutputStream tos = null;
            if (fileOutputStream != null && isConsoleOutputRequired) {
                tos = new TeeOutputStream(fileOutputStream, System.out);
                ExecuteStreamHandler streamHandler = new PumpStreamHandler(tos);
                executor.setStreamHandler(streamHandler);
            } else if (fileOutputStream != null) {
                ExecuteStreamHandler streamHandler = new PumpStreamHandler(fileOutputStream);
                executor.setStreamHandler(streamHandler);
            }
            if (timeWait > 0) {
                ExecuteWatchdog watchdog = new ExecuteWatchdog(timeWait);
                executor.setWatchdog(watchdog);
            }
            executor.setExitValues(errorExitValues);
            LoggingHandler.getLogger().info("aster run command line:\n" + commandLine);
            exitValue = executor.execute(commandLine);
            if (fileOutputStream != null) {
                fileOutputStream.flush();
                fileOutputStream.close();
            }
        } catch (ExecuteException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return exitValue;
    }

    /**
     * @return list of options
     */
    public List<String> getOptions() {
        return this.options;
    }

    /**
     * sets os command working directory
     * 
     * @param workingDirUri
     *            absolute uri of os command working directory
     */
    public void setWorkingDirUri(String workingDirUri) {
        if (FileProc.newFile(workingDirUri).exists()) {
            if (preDeleteWorkDir) {
                File[] fs = FileProc.newFile(workingDirUri).listFiles();
                for (int k = 0; k < fs.length; k++) {
                    fs[k].delete();
                }
                FileProc.newFile(workingDirUri).delete();
            }
        } else if (!FileProc.newFile(workingDirUri).mkdirs()) {
            throw (new IllegalStateException("unable to create directory: " + workingDirUri));
        }
        this.workingDirUri = workingDirUri;
    }

    public String getWorkingDirUri() {
        return this.workingDirUri;
    }

    public void setFileOutputStream(String fileUri) {
        try {
            this.fileOutputStream = new FileOutputStream(FileProc.newFile(fileUri));
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        }
    }

    public void setPreDeleteWorkDir(boolean preDeleteWorkDir) {
        this.preDeleteWorkDir = preDeleteWorkDir;
    }

    public void setErrorExitValues(int[] errorExitValues) {
        this.errorExitValues = errorExitValues;
    }

    public void setTimeWait(long timeWait) {
        this.timeWait = timeWait;
    }

    /**
     * @param isConsoleOutputRequired
     *            the isConsoleOutputRequired to set
     */
    public void setConsoleOutputRequired(boolean isConsoleOutputRequired) {
        this.isConsoleOutputRequired = isConsoleOutputRequired;
    }

}
