/**
 * This file is part of waster-0.1.
 *
 *    waster-0.1 is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    waster-0.1 is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with waster-0.1.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.pimeca.waster.util;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.net.URI;
import java.net.URISyntaxException;

/**
 * convenience class to wrap file processing
 * 
 * @author maurel
 * 
 */
public class FileProc {
	private static int bufferSize = 32 * 1204;

	private FileProc() {
	}

	public static File newFile(String uriString) {
		File f = null;
		try {
			URI uri = new URI(uriString);
			f = new File(uri);
		} catch (java.net.URISyntaxException use) {
			use.printStackTrace();
		}
		return f;
	}

	public static File newFile(String baseUri, String relativeUri) {
		File f = null;
		try {
			URI uri = new URI(baseUri + relativeUri);
			f = new File(uri);
			if (f != null) {
				if (!f.exists()) {
					if (f.getParentFile().mkdirs()) {
						f.createNewFile();
					}
				}
			}
		} catch (URISyntaxException use) {
			use.printStackTrace();
		} catch (IOException ioe) {
			ioe.printStackTrace();
		}
		return f;
	}

	public static void copyFile(File file, File newFile) {
		if (!file.exists()) {
			throw (new IllegalArgumentException("file missing: " + file.toString()));
		}
		try {
			if (!newFile.exists()) {
				if (newFile.getParentFile().mkdirs()) {
					newFile.createNewFile();
				}
			}
			BufferedInputStream is = new BufferedInputStream(new FileInputStream(file), bufferSize);
			BufferedOutputStream os = new BufferedOutputStream(new FileOutputStream(newFile), bufferSize);
			byte[] buff = new byte[bufferSize];
			int len;
			while ((len = is.read(buff)) > 0)
				os.write(buff, 0, len);
			is.close();
			os.close();
		} catch (IOException ioe) {
			ioe.printStackTrace();
		}
	}

	public static void copyFile(String srcUri, String tgtUri) {
		FileProc.copyFile(FileProc.newFile(srcUri), FileProc.newFile(tgtUri));
	}

	public static File createFile(String uri) {
		return FileProc.createFile(FileProc.newFile(uri));
	}

	public static File createFile(File file) {
		if (file == null) {
			throw (new NullPointerException("file is null"));
		}
		if (!file.exists()) {
			File parent = file.getParentFile();
			if (!parent.exists()) {
				boolean isDirCreated = parent.mkdirs();
				if (!isDirCreated) {
					throw (new IllegalStateException("unable to create dir: " + parent.getAbsolutePath()));
				}
			}
			try {
				if (!file.createNewFile()) {
					throw (new IllegalStateException("unable to create file: " + file.getAbsolutePath()));
				}
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
		return file;
	}

	public static void createFile(BufferedInputStream bis, String tgtUri) {
		File tgt = FileProc.createFile(FileProc.newFile(tgtUri));
		try {
			BufferedOutputStream bos = new BufferedOutputStream(new FileOutputStream(tgt), bufferSize);
			byte[] buff = new byte[bufferSize];
			int len;
			while ((len = bis.read(buff)) > 0)
				bos.write(buff, 0, len);
			bos.close();
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	public static String getString(File file) {
		String out = "";
		try {
			BufferedInputStream is = new BufferedInputStream(new FileInputStream(file));
			StringWriter sw = new StringWriter();
			int b;
			while ((b = is.read()) != -1) {
				sw.write(b);
			}
			is.close();
			out = sw.toString();
			sw.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
		return out;
	}

	public static void fillTextFile(File file, String content) {
		try {
			StringReader sr = new StringReader(content);
			BufferedOutputStream os = new BufferedOutputStream(new FileOutputStream(file));
			int b = 0;
			while ((b = sr.read()) != -1) {
				os.write(b);
			}
			os.close();
			sr.close();
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	public static void append(File file, String content) {
		FileProc.append(file, content, true);
	}

	public static void append(File file, String content, boolean isEndOfLineToAdd) {
		try {
			BufferedWriter out = new BufferedWriter(new FileWriter(file.getAbsolutePath(), true));
			content = (isEndOfLineToAdd) ? content + "\n" : content;
			out.write(content);
			out.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	public static File copyDirectory(File src, File tgt) {
		if (src.isDirectory()) {
			if (!tgt.exists()) {
				if (!tgt.mkdirs()) {
					throw (new IllegalStateException("unable to create tgt: " + tgt.getAbsolutePath()));
				}
			}
			String[] children = src.list();
			for (int i = 0; i < children.length; i++) {
				copyDirectory(new File(src, children[i]), new File(tgt, children[i]));
			}
		} else {
			FileProc.copyFile(src, tgt);
		}
		return tgt;
	}

	public static File copyDirectory(String srcUri, String tgtUri) {
		return FileProc.copyDirectory(FileProc.newFile(srcUri), FileProc.newFile(tgtUri));
	}

	public static void deleteDir(String uri) {
		if (FileProc.newFile(uri).isDirectory()) {
			if (!FileProc.newFile(uri).delete()) {
				File[] lf = FileProc.newFile(uri).listFiles();
				for (int i = 0; i < lf.length; i++) {
					if (lf[i].isFile()) {
						lf[i].delete();
					} else if (lf[i].isDirectory()) {
						FileProc.deleteDir(lf[i].toURI().toString());
					} else {
						throw (new IllegalStateException("unidentified object: " + lf[i]));
					}
				}
				FileProc.newFile(uri).delete();
			}
		} else {
			throw (new IllegalArgumentException("unable to delete: " + uri));
		}
	}

	public static File createDir(String uri) {
		File file = FileProc.newFile(uri);
		createDir(file);
		return file;
	}

	public static File createDir(File file) {
		if (file == null) {
			throw (new NullPointerException("file is null"));
		}
		if (!file.exists()) {
			File parent = file.getParentFile();
			if (!parent.exists()) {
				boolean isDirCreated = parent.mkdirs();
				if (!isDirCreated) {
					throw (new IllegalStateException("unable to create dir: " + parent.getAbsolutePath()));
				}
			}
			if (!file.mkdir()) {
				throw (new IllegalStateException("unable to create file: " + file.getAbsolutePath()));
			}
		}
		return file;
	}

}
